/**************************************************************************
   Filename: Menu.inl
Description:
  Implements in-game menus as a finite-state manager of nodes with poly-
  morphic behavior. Allows for (among other things) nested submenus in an
  arbitrary tree structure.
**************************************************************************/

template <typename T>
void Menu::Node::AddEdge(int neighborIndex, T checkCondition)
{
  m_Edges.emplace_back(neighborIndex, checkCondition);
}

template <typename T1, typename T2>
void Menu::Node::AddInteraction(T1 checkCondition, T2 execute)
{
  m_Interactions.emplace_back(checkCondition, execute);
}

template <typename BehaviorType>
void Menu::SetBehavior()
{
  m_Behavior = std::make_unique<BehaviorType>(*this);
  m_Behavior->Initialize();
  m_Behavior->Create();
}

template <typename BehaviorType>
BehaviorType& Menu::GetBehavior()
{
#ifdef _DEBUG
  if (!m_Behavior)
  {
    Error::RuntimeError("Tried to get behavior from menu, but it had none.");
  }
#endif

  BehaviorType* behavior = dynamic_cast<BehaviorType*>(m_Behavior.get());

#ifdef _DEBUG
  if (!behavior)
  {
    Error::RuntimeError("Requested wrong behavior type from menu.");
  }
#endif

  return *behavior;
}

/*!
*   Connects all nodes in a menu forward and backward using the provided functors as edges.
*   If wrap is true, first and last node will be connected (otherwise they won't)
*/
template <typename T1, typename T2>
void Menu::ConnectNodes(T1 forward, T2 backward, bool wrap)
{
    // Connect the first and last nodes to their successor/predecessor, respectively
  (*this)[0].AddEdge(1, forward);
  (*this)[(*this).Size() - 1].AddEdge(static_cast<int>((*this).Size()) - 2, backward);
  
    // Connect all the nodes that have both a predecessor and a successor
  for (int i = 1; i < static_cast<int>((*this).Size()) - 1; ++i)
  {
    (*this)[i].AddEdge(i + 1, forward);
    (*this)[i].AddEdge(i - 1, backward);
  }

    // If wrap is true, connect the first and last node to each other
  if (wrap)
  {
    (*this)[0].AddEdge(static_cast<int>((*this).Size()) - 1, backward);
    (*this)[(*this).Size() - 1].AddEdge(0, forward);
  }
}
