/**************************************************************************
   Filename: Menu.h
Description:
  Implements in-game menus as a finite-state manager of nodes with poly-
  morphic behavior. Allows for (among other things) nested submenus in an
  arbitrary tree structure.
**************************************************************************/

#pragma once

class Menu : public BaseBehavior
{
  // Public Classes:
public:
  class Node : public BaseBehavior
  {
  private:
    /*!
    *   Nodes are connected by edges - they allow a user to travel from the current node to a neighbor
    */
    struct Edge
    {
      template <typename Condition>
      Edge(int neighbor, Condition condition): neighbor(neighbor), condition(condition) {}

      std::function<bool()> condition;
      int neighbor; // Index of node edge connects to
    }; // struct Edge

  public:
    /*!
    *   Check for user interaction (only runs on the current node)
    */
    void CheckInteractions();

    /*!
    *   Check for user traveling from current node to a neighbor
    */
    int CheckEdges() const;

    void SetText(std::string const& text) { m_Text = text; }

    std::string const& GetText() const { return m_Text; }

    /*!
    *   Get a reference to text settings (directly editing is fine)
    */
    //Text::Settings& GetTextSettings() { return m_TextSettings; }

    void SetPos(Vec<2, float> const& pos) { m_Pos = pos; }

    Vec<2, float> GetPos() const { return m_Pos; }

    template <typename T, typename... Args>
    T& AddObject(Args... args)
    {
      m_Objects.emplace_back(std::make_shared<T>(std::forward<Args>(args)...));
      return *static_cast<T*>(m_Objects.back().get());
    }

    BaseBehavior* GetObject(std::string const& name);

    BaseBehavior* GetObject(size_t index);

    template <typename T>
    void AddEdge(int neighborIndex, T check);

    template <typename T1, typename T2>
    void AddInteraction(T1 checkCondition, T2 execute);

    bool Initialize() override;

    void EarlyUpdate() override;

    void Update(float dt) override;

    void Render() override;

    void LateUpdate() override;

    void Read(SerialData& data) override;

    void Write(SerialData& data) const override;

  private:
    std::string m_Text;
    std::vector<std::shared_ptr<BaseBehavior>> m_Objects;
    std::vector<Utils::ConditionalAction> m_Interactions;
    std::vector<Edge> m_Edges;
    Vec<2, float> m_Pos; // In world units, directly to viewport (ignores camera/world position)
  }; // class Node

  // Constructors:
public:
  Menu(std::string const& name = "") : BaseBehavior(name) {}

  // Iterators:
public:
  auto begin() { return m_Nodes.begin(); }
  auto end() { return m_Nodes.end(); }

  auto begin() const { return m_Nodes.begin(); }
  auto end() const { return m_Nodes.end(); }

  // Operators:
public:
  Node& operator[](int index) { return m_Nodes[index]; }
  Node const& operator[](int index) const { return m_Nodes[index]; }

  // Public Static Functions:
public:
  static Menu& GetInstance() { return *m_Instance; }

  // Public Functions:
public:
  bool Empty() const { return m_Nodes.empty(); }

  size_t Size() const { return m_Nodes.size(); }

  /*!
  *   Set the menu's node count
  */
  void Resize(size_t size) { m_Nodes.resize(size); }

  Node& EmplaceBack() { return m_Nodes.emplace_back(); }

  Node& Back() { return m_Nodes.back(); }

  Node const& Back() const { return m_Nodes.back(); }

  void SetCurrent(int index);

  int GetCurrent() const { return m_Current; }

  void ToggleUpdateAll(bool state) { m_UpdateAll = state; }

  void ToggleRenderAll(bool state) { m_RenderAll = state; }

  Text::Settings& GetTextSettings() { return m_TextSettings; }

  void SetSelectedPalette(int paletteIndex) { m_SelectetPaletteIndex = paletteIndex; }

  template <typename BehaviorType>
  void SetBehavior();

  template <typename BehaviorType>
  BehaviorType& GetBehavior();
    
  /*!
  *   Connects all nodes in a menu forward and backward using the provided functors as edges.
  *   If wrap is true, first and last node will be connected (otherwise they won't)
  */
  template <typename T1, typename T2>
  void ConnectNodes(T1 forward, T2 backward, bool wrap = true);

  /*!
  *   Connects all nodes in a menu using the provided input triggers as edges
  */
  void ConnectNodes_Trigger(Input::Name increment, Input::Name decrement, bool wrap = true);

  /*!
  *   Clears all the nodes and sets current index to 0
  */
  void Clear();

  /*!
  *   Resets all the menu data (name, behavior, nodes)
  */
  void Reset();

  bool Initialize() override;

  void EarlyUpdate() override;

  void Update(float dt) override;

  void Render() override;

  void LateUpdate() override;

  void Read(SerialData& data) override;

  void Write(SerialData& data) const override;

  // Private Static Members:
private:
  static std::unique_ptr<Menu> m_Instance;

private:
  Text::Settings m_TextSettings;
  std::vector<Node> m_Nodes;
  std::unique_ptr<BehaviorMenu> m_Behavior;
  int m_Current = 0;        // Index of selected node
  int m_SelectetPaletteIndex = -1; // If >= 0, selected menu node text uses this palette index
  bool m_UpdateAll = false; // Call update all nodes (as opposed to just the selected one)
  bool m_RenderAll = true;  // Call render on all nodes (as opposed to just the selected one)
}; // class Menu

#include "Menu.inl"