/**************************************************************************
   Filename: Menu.cpp
Description:
  Implements in-game menus as a finite-state manager of nodes with poly-
  morphic behavior. Allows for (among other things) nested submenus in an
  arbitrary tree structure.
**************************************************************************/

#include "pch.h"

std::unique_ptr<Menu> Menu::m_Instance = std::make_unique<Menu>();

/*!
*   Check for user interaction (only runs on the current node)
*/
void Menu::Node::CheckInteractions()
{
  for (auto& interaction : m_Interactions)
  {
    interaction.Run();
  }
}

/*!
*   Check for user traveling from current node to a neighbor
*/
int Menu::Node::CheckEdges() const
{
  for (auto const& edge : m_Edges)
  {
    if (edge.condition())
    {
      return edge.neighbor;
    }
  }

  return -1; // Not traveling to any neighbor
}

BaseBehavior* Menu::Node::GetObject(std::string const& name)
{
  for (auto& object : m_Objects)
  {
    if (object->Name() == name)
    {
      return object.get();
    }
  }

  return nullptr;
}

BaseBehavior* Menu::Node::GetObject(size_t index)
{
  if (index >= 0 && index < m_Objects.size())
  {
    return m_Objects[index].get();
  }

  return nullptr;
}

bool Menu::Node::Initialize()
{
  //m_TextSettings.font.Initialize();

  for (auto& object : m_Objects)
  {
    if (!object->Initialize())
    {
      return false;
    }
  }

  return true;
}

void Menu::Node::EarlyUpdate()
{
  for (auto& object : m_Objects)
  {
    object->EarlyUpdate();
  }
}

void Menu::Node::Update(float dt)
{
  for (auto& object : m_Objects)
  {
    object->Update(dt);
  }
}

void Menu::Node::Render()
{
  if (m_Text != "")
  {
    //Text::ReadSettings(m_TextSettings);
    Text::Print(m_Text, m_Pos);
  }

  for (auto& object : m_Objects)
  {
    object->Render();
  }
}

void Menu::Node::LateUpdate()
{
  for (auto& object : m_Objects)
  {
    object->LateUpdate();
  }
}

void Menu::Node::Read(SerialData& data)
{
  if (data.Contains("Objects"))
  {
    SerialData& objectsData = data.Get<SerialData>("Objects");

    for (auto& object : m_Objects)
    {
      if (objectsData.Contains(object->Name()))
      {
        object->Read(objectsData.Get<SerialData>(object->Name()));
      }
    }
  }
}

void Menu::Node::Write(SerialData& data) const
{
  SerialData objectsData;

  for (auto& object : m_Objects)
  {
    SerialData objectData;
    object->Write(objectData);
    objectsData.Insert(object->Name(), objectData);
  }

  data.Insert("Objects", objectsData);
}

void Menu::SetCurrent(int index)
{ 
#ifdef _DEBUG
  if (index < 0 || index >= Size())
  {
    Error::RuntimeError("Menu index out of range.");
  }
#endif

  m_Current = index;
}

void Menu::ConnectNodes_Trigger(Input::Name increment, Input::Name decrement, bool wrap)
{
  if (Size() > 1)
  {
    Input* input = Engine::GetSystem<Input>();

    ConnectNodes(
      [input]() { return input->IsTriggered(Input::DOWN); },
      [input]() { return input->IsTriggered(Input::UP);   }
    );
  }
}

/*!
*   Clears all the nodes and sets current index to 0
*/
void Menu::Clear()
{
  m_Nodes.clear();
  m_Current = 0;
}

/*!
*   Resets all the menu data (name, behavior, objects, nodes)
*/
void Menu::Reset()
{
  if (Name().empty() || !m_Nodes.empty() || m_Behavior)
  {
    SIGNAL_EVENT(MenuDestroyed, Name());
    m_Behavior.reset();
    m_Nodes.clear();
    m_Current = 0;
    SetName("");
  }
}

bool Menu::Initialize()
{
  if (m_Behavior && !m_Behavior->Initialize())
  {
    return false;
  }

  for (auto& node : m_Nodes)
  {
    if (!node.Initialize())
    {
      return false;
    }
  }

  return true;
}

void Menu::EarlyUpdate()
{
  if (m_Behavior)
  {
    m_Behavior->EarlyUpdate();
  }
    
  if (!m_Nodes.empty())
  {
    if (m_UpdateAll)
    {
      for (auto& node : m_Nodes)
      {
        node.EarlyUpdate();
      }
    }

    else
    {
      m_Nodes[m_Current].EarlyUpdate();
    }
  }
}

void Menu::Update(float dt)
{
  if (m_Behavior)
  {
    m_Behavior->Update(dt);
  }

  if (!m_Nodes.empty())
  {
    if (m_UpdateAll)
    {
      for (auto& node : m_Nodes)
      {
        node.Update(dt);
      }
    }

    else
    {
      m_Nodes[m_Current].Update(dt);
    }

    m_Nodes[m_Current].CheckInteractions();

    int neighbor = m_Nodes[m_Current].CheckEdges();

      // Edge was traversed
    if (neighbor >= 0)
    {
      m_Current = neighbor;
    }
  }
}

void Menu::Render()
{
  if (m_Behavior)
  {
    m_Behavior->Render();
  }

  if (!m_Nodes.empty())
  {
    Text::ReadSettings(m_TextSettings);

    if (m_RenderAll)
    {
      for (auto& node : m_Nodes)
      {
        if (&node == &m_Nodes[m_Current] && m_SelectetPaletteIndex >= 0)
        {
          Text::SetPaletteIndex(m_SelectetPaletteIndex);
        }

        node.Render();

        if (&node == &m_Nodes[m_Current] && m_SelectetPaletteIndex >= 0)
        {
          Text::SetPaletteIndex(m_TextSettings.paletteIndex);
        }
      }
    }

    else
    {
      if (m_SelectetPaletteIndex >= 0)
      {
        Text::SetPaletteIndex(m_SelectetPaletteIndex);
      }

      m_Nodes[m_Current].Render();
    }
  }
}

void Menu::LateUpdate()
{
  if (m_Behavior)
  {
    m_Behavior->LateUpdate();
  }

  if (!m_Nodes.empty())
  {
    if (m_UpdateAll)
    {
      for (auto& node : m_Nodes)
      {
        node.LateUpdate();
      }
    }

    else
    {
      m_Nodes[m_Current].LateUpdate();
    }
  }
}

void Menu::Read(SerialData& data)
{
  if (data.Contains("Type"))
  {
    menuRegistry.Call(data.Get<std::string>("Type"), *this);
    m_Behavior->Read(data);
  }

  //if (data.Contains("Nodes"))
  //{
  //  SerialData& nodeData = data.Get<SerialData>("Nodes");

  //    // Get the number of nodes in the menu and set the size
  //  assert(nodeData.Contains("Size") && "Tried to read menu node data, but size was not specified.");
  //  Clear();
  //  Resize(nodeData.Get<int>("Size"));

  //    // Read data for the individual nodes
  //  assert(nodeData.Contains("Nodes") && "Tried to read menu node data, but it was missing the node list.");
  //  SerialData::Array& nodeList = nodeData.Get<SerialData::Array>("Nodes");
  //  int index = 0;

  //  for (auto& node : nodeList)
  //  {
  //    m_Nodes[index].Read(node->Get<SerialData>());
  //  }
  //}
}

void Menu::Write(SerialData& data) const
{
    // Write the menu's type to data object
  SerialData menuData;

    // Write object data (all objects stored in one SerialData object)
  SerialData objectsData;

  if (m_Behavior)
  {
    SerialData behaviorData;
    behaviorData.Insert("Type", Utils::GetTypeName(*m_Behavior));
    m_Behavior->Write(behaviorData);
    data.Insert("Behavior", behaviorData);
  }

  menuData.Insert("Objects", objectsData);

    // Write node data (not currently in use)
  SerialData nodesData;
  nodesData.Insert("Size", static_cast<int>(Size()));

    // Write data for node list
  SerialData::Array nodesList;

  for (auto& node : m_Nodes)
  {
    SerialData nodeData;
    node.Write(nodeData);
    nodesList.EmplaceBack<SerialData>(nodeData);
  }

    // Insert nodes list into node data, node data into menu data, menu data into given data object
  nodesData.Insert("Nodes", nodesList);
  menuData.Insert("Nodes", nodesData);
  data.Insert("Menu", menuData);
}
